/*
 *	Generic parts
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/llc.h>
#include <net/llc.h>
#include <net/stp.h>
#include <linux/version.h>

#include "br_private.h"

int (*br_should_route_hook)(struct sk_buff *skb);

static const struct stp_proto br_stp_proto = {
	.rcv	= br_stp_rcv,
};

static struct pernet_operations br_net_ops = {
	.exit	= br_net_exit,
};

static struct proc_dir_entry * dhcp_server_interface;
char g_szdhcp_server_interface[32] = {0};

#ifdef CONFIG_DHCP_MON
int g_dhcpmon_pid = 0;
struct sock *netlink_dhcpmon_sock;
#endif

static int dhcp_server_interface_write_proc(struct file *file, const char *buffer, unsigned long count, void *data)
{
	unsigned long len = count;
	unsigned long iVal;

	if(len > 32) {
		len = 32;
	}
	memset(g_szdhcp_server_interface, 0x00, sizeof(g_szdhcp_server_interface));
	if(copy_from_user(g_szdhcp_server_interface, buffer, len)) {
		return -EFAULT;
	}
    
    printk("g_szdhcp_server_interface = %s\n",g_szdhcp_server_interface);
	
	return len;
}
static int dhcp_server_interface_read_proc(char *buf, char **start, off_t off, int count, int *eof, void *data)
{
	int index = 0;

	index += sprintf(buf+index, "%s\n", g_szdhcp_server_interface);
	if(index <= off+count) {
		*eof = 1;
	}
	*start = buf + off;
	index -= off;
	if(index>count) {
		index = count;
	}
	if(index < 0) {
		index = 0;
	}

	return index;
}

#ifdef CONFIG_DHCP_MON
static int dhcpmon_pid_write_proc( struct file *filp, const char __user *buf,unsigned long len, void *data )
{
	
	int ret;
	ret = sscanf(buf, "%d", &g_dhcpmon_pid);
	return len;
}

static int dhcpmon_pid_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	int ret;
	ret = snprintf(buf, len, "%d", g_dhcpmon_pid);
	return ret;
}
#endif

static int __init br_init(void)
{
	int err;
	struct proc_dir_entry *proc_pid_dhcpmon;

	err = stp_proto_register(&br_stp_proto);
	if (err < 0) {
		pr_err("bridge: can't register sap for STP\n");
		return err;
	}

	err = br_fdb_init();
	if (err)
		goto err_out;

	err = register_pernet_subsys(&br_net_ops);
	if (err)
		goto err_out1;

	err = br_netfilter_init();
	if (err)
		goto err_out2;

	err = register_netdevice_notifier(&br_device_notifier);
	if (err)
		goto err_out3;

	err = br_netlink_init();
	if (err)
		goto err_out4;

	brioctl_set(br_ioctl_deviceless_stub);

#if defined(CONFIG_ATM_LANE) || defined(CONFIG_ATM_LANE_MODULE)
	br_fdb_test_addr_hook = br_fdb_test_addr;
#endif
    if((dhcp_server_interface = create_proc_entry( "dhcp_server_interface", 0644, NULL)) != NULL)
    {
        dhcp_server_interface->read_proc = dhcp_server_interface_read_proc;
        dhcp_server_interface->write_proc = dhcp_server_interface_write_proc;
    }
#ifdef CONFIG_DHCP_MON
	proc_pid_dhcpmon = create_proc_entry( "dhcpmon_pid", 0644, NULL);
	if(NULL != proc_pid_dhcpmon)
	{
		proc_pid_dhcpmon->read_proc = dhcpmon_pid_read_proc;
		proc_pid_dhcpmon->write_proc = dhcpmon_pid_write_proc;
	}
	
#if(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,22))
	netlink_dhcpmon_sock = netlink_kernel_create(NETLINK_DHCP_TBS, 0, NULL, THIS_MODULE);
#elif(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24))
	netlink_dhcpmon_sock = netlink_kernel_create(NETLINK_DHCP_TBS, 0, NULL, NULL, THIS_MODULE);
#else
	netlink_dhcpmon_sock = netlink_kernel_create(&init_net, NETLINK_DHCP_TBS, 0, NULL, NULL, THIS_MODULE);
#endif
#endif

	return 0;
err_out4:
	unregister_netdevice_notifier(&br_device_notifier);
err_out3:
	br_netfilter_fini();
err_out2:
	unregister_pernet_subsys(&br_net_ops);
err_out1:
	br_fdb_fini();
err_out:
	stp_proto_unregister(&br_stp_proto);
	return err;
}

static void __exit br_deinit(void)
{
	stp_proto_unregister(&br_stp_proto);

	br_netlink_fini();
	unregister_netdevice_notifier(&br_device_notifier);
	brioctl_set(NULL);

	unregister_pernet_subsys(&br_net_ops);

	rcu_barrier(); /* Wait for completion of call_rcu()'s */

	br_netfilter_fini();
#if defined(CONFIG_ATM_LANE) || defined(CONFIG_ATM_LANE_MODULE)
	br_fdb_test_addr_hook = NULL;
#endif

	br_fdb_fini();
}

EXPORT_SYMBOL(br_should_route_hook);

module_init(br_init)
module_exit(br_deinit)
MODULE_LICENSE("GPL");
MODULE_VERSION(BR_VERSION);
